/** @file   soundsource.cpp
 * @brief   Implementation of SoundSource - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#include "soundsource.h"

namespace eng2d {

///
/// Static members, constants and datatypes
/// =======================================

float SoundSource::maxDistance = 400;
float SoundSource::maxPanSwing = 80;




///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
SoundSource::SoundSource( const std::string& soundname, const Vec2D& pos ) :
  Sample(),
  m_name( soundname ),
  m_pos( pos ),
  m_loop( true ),
  m_voiceIndex( -1 ),
  m_active( false ),
  m_loudness( 128 )
{
}



/** Destructor
 */
SoundSource::~SoundSource()
{
  this->turnOff();
}




///
/// Public methods
/// ==============

/** Sets the position of this sample.
 */
void SoundSource::position( const Vec2D& rP )
{
  m_pos = rP;
}



/** Sets the loop-flag
 */
void SoundSource::loop( bool l )
{
  m_loop = l;
}



/** Sets the loudness
 */
void SoundSource::loudness( int loud )
{
  m_loudness = loud;
}


/** Starts the audio playback
 */
void SoundSource::turnOn()
{
  SAMPLE* pS = this->al_SAMPLE();
  if ( pS == 0 || m_active == true ) {
    return;
  }
  
  m_voiceIndex = play_sample( pS, 0, 128, 1000, static_cast<int>( m_loop ) );
  m_voiceReleased = false;
  if ( m_voiceIndex >= 0 ) {
    m_active = true;
  }
}



/** Turns this sound off
 */
void SoundSource::turnOff()
{
  if ( m_active == false ) {
    return;
  }
  
  m_active = false;
  if ( m_voiceIndex >= 0 ) {
    deallocate_voice( m_voiceIndex );
    m_voiceIndex = -1;
  }
  m_voiceReleased = true;
}



/** Updates this sound source
 */
void SoundSource::update( const Vec2D& rListener )
{
  // If this sound source is not active, we don't have anything to do.
  if ( this->active() == false ) {
    return;
  }
  
  
  
  // Calculate our distance from the listener.
  Vec2D distV( rListener );
  distV -= m_pos;
  float d = distV.length();
  
  
  // Adjust the volume and panning based on the distance and direction
  // the listener is when compared to our position.
  if ( d > maxDistance ) {
    // The listener is so far away from the listener that we don't
    // need to keep the soundcard's voice allocated for us.
    if ( m_voiceIndex >= 0 ) {
      deallocate_voice( m_voiceIndex );
      m_voiceIndex = -1;
      m_voiceReleased = true;
    }
    
    
  } else {
    // If the voice we used have been deallocated, we need to allocate
    // new voice for this sound.
    // Calculate new volume and panning values.
    int vol = this->calculateVolume( d );
    int pan = this->calculatePanning( rListener );
    if ( m_voiceIndex < 0 ) {
      m_voiceIndex = play_sample( this->al_SAMPLE(), vol, pan, 1000,
                                  static_cast<int>( m_loop) );
      m_voiceReleased = false;
      
    } else {
     // Adjust the voice attributes.
      voice_set_volume( m_voiceIndex, vol );
      voice_set_pan( m_voiceIndex, pan );
    }
  }
}




///
/// Getter methods
/// ==============

/** Returns the position of this sound source
 */
Vec2D SoundSource::position() const
{
  return m_pos;
}



/** Returns the name of this sound source
 */
const std::string& SoundSource::name() const
{
  return m_name;
}



/** Tells if this sound source is active.
 */
bool SoundSource::active() const
{
  return m_active;
}



/** Returns the loudness of this soundsource
 */
int SoundSource::loudness() const
{
  return m_loudness;
}




///
/// Private or Protected methods
/// ============================

/** Calculates the volume
 */
int SoundSource::calculateVolume( float dist ) const
{
  if ( dist >= maxDistance ) {
    return 0;
  }
  
  float loud = static_cast<float>( this->loudness() );
  float k = loud / (-maxDistance);

  int vol = static_cast<int>( dist * k );
  vol += this->loudness();
  return vol;
}



/** Calculates the panning value when listener is at given coordinates
 */
int SoundSource::calculatePanning( const Vec2D& rListener ) const
{
  // Calculate the minimum and maximum pan-values. The 1.28 in these
  // equations come from  128 / 100. This means the number of pan-units
  // represented by one percentage.
  float minPan = 128 - (1.28 * maxPanSwing);
  float maxPan = 128 + (1.28 * maxPanSwing);
  
  // Is the x-difference is too big.
  float deltaX = m_pos.x() - rListener.x();
  if ( deltaX < -(maxDistance/2) ) {
    return static_cast<int>( minPan );
    
  } else if ( deltaX > (maxDistance/2) ) {
    return static_cast<int>( maxPan );
    
  }
  
  // Calculate the equations for the panning graph.
  float k = (maxPan - minPan) / maxDistance;
  int pan = static_cast<int>( k * deltaX );
  pan += 128;
  return pan;
}


} // end of namespace
